/**
  Copyright (c) 2010 Freescale Semiconductor
  
  \file  	  Graphics.h
  \brief	  This is the Graphics Header Driver File
  \brief	  Provides functionality to mirror, rotate, sub-scale and copy images
  \author	  Freescale Semiconductor
  \author	  Automotive Systems Solutions Engineering
  \author	  IM, b06623
  \version	  2.0
  \revision	  $Revision: 245 $
  \date  	  $Date: 2011-09-22 17:44:03 -0500 (Thu, 22 Sep 2011) $  
  
  * History:  20/August/2008 - Initial Version (IM)
              13/May/2009 - MISRA Check. Release (IM) 
              25/May/2010 - Update to improve performance. Removed usage of cfg structures and
              				functions from the DMA (IM)
              April-2010  - Support DCU3 bpp modes (IM)
              Ago-2010    - Support DCU3 tile, rle, partial blank area prototypes and modified typedefs (IM)			

* Copyright (c) 2010, Freescale, Inc.  All rights reserved.
*
*
* No part of this document must be reproduced in any form - including copied,
* transcribed, printed or by any electronic means - without specific written
* permission from Freescale Semiconductor.
*

*/


#ifndef 	_GRAPHICS_H_
#define  	_GRAPHICS_H_	 

#include "..\HAL\DMA_drv.h"

/************************** Data Types ***************************************/

#ifndef GRAPHICS_BITMAP_ALIGN
#define GRAPHICS_BITMAP_ALIGN _Pragma("data_alignment=64")
#endif

typedef  enum
{
	GRAPHICS_CODING_RAWINT = 0,
	GRAPHICS_CODING_RLEINT,
	GRAPHICS_CODING_TILE,
	GRAPHICS_CODING_RLEEXT
} Graphics_Coding_t;

#define GRAPHICS_CTX0	0
#define GRAPHICS_CTX1	1

typedef  void  ( * Graphics_CallbackType )( void  );



#ifdef DCU4
#define  	GRAPHICS_YCBCR422	   (14u)
#define  	GRAPHICS_APAL8		   (13u)
#endif
#define  	GRAPHICS_4444ARGB	   (12u)
#define  	GRAPHICS_1555ARGB	   (11u)
#define  	GRAPHICS_L8BPP		   (10u)
#define  	GRAPHICS_L4BPP		   (9u)
#define  	GRAPHICS_T8BPP		   (8u)
#define  	GRAPHICS_T4BPP		   (7u)
#define 	GRAPHICS_32BPP		   (6u)
#define 	GRAPHICS_24BPP		   (5u)
#define  	GRAPHICS_565RGB		   (4u)
#define  	GRAPHICS_8BPP		   (3u)
#define  	GRAPHICS_4BPP		   (2u)
#define  	GRAPHICS_2BPP		   (1u)
#define  	GRAPHICS_1BPP		   (0u)

typedef  enum
{
	GRAPHICS_ERROR_OK, 
	GRAPHICS_ERROR_FAIL
} Graphics_ErrorType;

typedef  struct
{
    uint32_t  address;			/*	< Address in memory >				  */
    uint16_t  width;			/*	< W Dimension of sprite in pixels >	  */
    uint16_t  height;			/*	< H Dimension of sprite in pixels >	  */
    uint32_t  *CLUT;			/*	< Source address of CLUT >			  */
    uint16_t  CLUTsize;  		/*	< Number of bytes in CLUT > 		  */
    uint16_t  CLUTOffset;		/*	< CLUT offset in palette memory >	  */
    uint8_t   BPP;					/*	< Bits per pixel coding >			*/
    uint8_t   initialAlpha;  		/*	< Initial alpha value >  			*/
    uint32_t  initialChromaMax;		/*	< Initial chroma setting			*/
    uint32_t  initialChromaMin;		/*	< Initial chroma setting          	*/
    Graphics_Coding_t   coding;		/*	< If RLE is used or not. (0 = not used) >	*/
    int16_t	x;						/*	< Initial position in x >	  */
    int16_t	y;						/*	< Initial position in y >	  */
    void	*ext;					/*  < To extend this structure to hold more data */
} Graphics_Object_t;


/************************ Public Functions ***********************************/

void Graphics_Init(void);
void Graphics_CopyGraphicObject(const Graphics_Object_t *source, Graphics_Object_t *target);

Graphics_ErrorType 	Graphics_BlankArea
(
	uint8_t eDMAChannel, 
	const Graphics_Object_t*  g_object, 
	uint32_t value, 
	Graphics_CallbackType callback
);

Graphics_ErrorType Graphics_BlankArea_Rect
(
	uint8_t eDMAChannel, 
	const Graphics_Object_t*  g_object,
	uint16_t xi,
	uint16_t yi,
	uint16_t xf,
	uint16_t yf, 
	uint32_t value, 
	Graphics_CallbackType callback
);

Graphics_ErrorType 	Graphics_SubCopy
(
	uint8_t  eDMAChannel,
	Graphics_Object_t*  Source, 
	Graphics_Object_t*  Target,
	uint16_t xi,
	uint16_t yi,
	uint16_t xf,
	uint16_t yf,
	Graphics_CallbackType callback
);

Graphics_ErrorType Graphics_Paste
(
	uint8_t  eDMAChannel,
	Graphics_Object_t*  Source, 
	Graphics_Object_t*  Target,
	uint16_t xi,
	uint16_t yi,
	Graphics_CallbackType callback
);

Graphics_ErrorType RLEData_Paste
(
	uint8_t  eDMAChannel,
	Graphics_Object_t*  Source, 
	Graphics_Object_t*  Target,
	uint32_t Dsize,
	Graphics_CallbackType callback
);

Graphics_ErrorType 	Graphics_Downscale
(
	uint8_t eDMAChannel,
	Graphics_Object_t* Source, 
	Graphics_Object_t* Target,
	uint8_t Factor_W, 
	uint8_t	Factor_H,
	Graphics_CallbackType callback
);

Graphics_ErrorType 	Graphics_Rotate90( uint8_t  eDMAChannel, Graphics_Object_t* Source, Graphics_Object_t*  Target, Graphics_CallbackType callback);
Graphics_ErrorType 	Graphics_Rotate180( uint8_t	eDMAChannel, Graphics_Object_t* Source, Graphics_Object_t* Target, Graphics_CallbackType callback);
Graphics_ErrorType 	Graphics_Rotate270(uint8_t eDMAChannel, Graphics_Object_t* Source, Graphics_Object_t* Target, Graphics_CallbackType callback);
Graphics_ErrorType 	Graphics_MirrorV(uint8_t eDMAChannel, Graphics_Object_t * Source, Graphics_Object_t * Target, Graphics_CallbackType callback);
Graphics_ErrorType 	Graphics_MirrorH( uint8_t  eDMAChannel, Graphics_Object_t*  Source, Graphics_Object_t*  Target, Graphics_CallbackType callback);
Graphics_ErrorType 	Graphics_Rotate90MirrorV( uint8_t  eDMAChannel, Graphics_Object_t*  Source, Graphics_Object_t*  Target, Graphics_CallbackType callback);
Graphics_ErrorType 	Graphics_Rotate90MirrorH( uint8_t  eDMAChannel, Graphics_Object_t*  Source, Graphics_Object_t*  Target, Graphics_CallbackType callback);
void 				Graphics_Decode8bits(uint8_t *buf, const uint8_t *compressed, int32_t len);
void 				Graphics_Decode8bitsDMA(uint8_t eDMAChannel, uint8_t *buf, const uint8_t *compressed, int32_t len);

void Graphics_RLEDMA
(
	uint8_t eDMAChannel,
	uint8_t eDMAChannel0, 
	Graphics_Object_t* src,
	Graphics_Object_t* dest,
	Graphics_CallbackType callback
);

/******************* Public Macros Function-alike *****************************/

#define Graphics_GetTileW(ext)		(((uint16_t *)(ext))[0])
#define Graphics_GetTileH(ext)		(((uint16_t *)(ext))[1])
#define Graphics_GetCompSize(ext)	(((uint32_t *)(ext))[0])


/********************** Private Data, DO NOT MODIFY ***************************************************************/

/* libsys.a abs funciton prototype to comply MISRA */
extern int abs( int  x );

#endif
